#!/bin/sh
# Copyright (C) 2013,2014 Tru-Test Ltd.  All rights reserved.

#
# This part is run in the background to allow startup.sh to finish.

POWER_SOURCE="UNKNOWN"
# update.sh use 'FORCE_REBOOT' as flag to reboot.
FORCE_REBOOT=0
# Go into standby for 2 hours
BATTERY_STANDBY_PERIOD_SECONDS=7200
# Wake up every 6 minutes (0.1 hours) during standby to check battery voltage
BATTERY_NAP_PERIOD_SECONDS=360

echo BATTERY_NAP_PERIOD_SECONDS: $BATTERY_NAP_PERIOD_SECONDS
echo BATTERY_STANDBY_PERIOD_SECONDS: $BATTERY_STANDBY_PERIOD_SECONDS

detect_power() {
    # Find out what power we are on
    STATUS=$(i2cget -y 1 0x09 0x04)
    # Mask for the WALL sense
    let "EXT_PWR_SOURCE=$STATUS & 0x80"
    if [ "$EXT_PWR_SOURCE" -eq 128 ]; then
        POWER_SOURCE="AC"
    else
        POWER_SOURCE="BATTERY"
    fi
    echo "Power source:" $POWER_SOURCE
}

configure_wake_interupts() {
    # Wake on LT4156 IRQ (writing also clears any old ints)
    i2cset -y 1 0x09 0x06 0x20

    # Stop the keyboard generating interupts
    i2cset -y 0 0x34 0x4e 0x00
    # Clear any pending keyboad interrupts (write one to clear)
    i2cset -y 0 0x34 0x01 0x01
}

# Read battery voltage from STC3100
wait_for_eoc_and_check_battery_voltage()
{
        echo Checking battery voltage.
	# A reading is ready every 2 seconds.  It alternates between voltage and temperature readings.
	# We need to check for reading ready to make sure the STC3100 is not in under-voltage lockout.
	# This could cause a 2 second delay, but in reality, a reading is normally always ready,
	# unless you try and take a reading within 2 seconds of a previous read of the status
	# register.  Reading the status register clears the vtmeoc flag.

	# 50x0.1 = 5 sec timeout for ready.
	let raw_adc=0;
	for i in `seq 50`
	do
	    let status=`i2cget -y 1 0x70 0x01`
	    # echo "STM3100 Status: " $status
	    let vtmeoc=$status\&8
	    let pordet=$status\&0x10
	    # echo "VTM_EOC: " $vtmeoc
            if [ $pordet -eq $((0x10)) ];
            then
                echo "PORDET: " $pordet
                i2cset -y 1 0x70 0x00 0x10
                i2cset -y 1 0x70 0x01 0x00
	    elif [ $vtmeoc -eq 8 ];
	    then
		# echo STM3100 Reading ready.
                let raw_adc=`i2cget -y 1 0x70 0x08 w`
                if [ $raw_adc -gt 0 ];
                then
                    break
                fi
	    fi
	    usleep 100000
	done

        # Return 0 mV if not VTM EOC detected.
	if [ $vtmeoc -eq 8 ]
	then
	    let raw_adc=`i2cget -y 1 0x70 0x08 w`
	fi

	# echo "Raw ADC: " $raw_adc
	let millivolts=$raw_adc\*244/100
	echo "Battery: " $millivolts mV
}

# This one does not wait for VTM_EOC.
fast_check_battery_voltage()
{
    let raw_millivolts=`i2cget -y 1 0x70 0x08 w`
    let millivolts=$raw_adc\*244/100
}

suspend_with_rtc() {
    echo Doing suspend_with_rtc...
    NOW_TIME=$(date +%s)
    STANDBY_FINISHED_TIME=$(expr $NOW_TIME + $BATTERY_STANDBY_PERIOD_SECONDS)

    # Default behavior (if not woken up by external power of a key press) is to power down
    # once the standby period is complete.
    LOOPING=0

    while [ $NOW_TIME -lt $STANDBY_FINISHED_TIME ]
    do
        wait_for_eoc_and_check_battery_voltage
        if [ $millivolts -lt 3000 ]
        then
            echo Battery voltage low.
            break
        fi

	check_battery_soc
        # 1000 mA.hr of 10 A.hr is 10% SOC.
        # IE one bar displayed in PolarisQtApp icon.
	if [ $battery_soc_mA_hour -lt 1000 ]
        then
            echo Battery SOC low.
            break
        fi

	# In case we miss the external power interrupt, do an extra check every 6 minute nap.
        detect_power
        if [ "$POWER_SOURCE" == "AC" ]
        then
            LOOPING=1
            break
        fi

        log_battery before_suspend_with_rtc
        # make sure log message is sync'ed.
        sync

	pre_sleep_raw_soc=$raw_soc
        configure_wake_interupts

        NOW_TIME=$(date +%s)
        SLEEP_START_TIME=$NOW_TIME
        NAP_END_TIME=$(expr $NOW_TIME + $BATTERY_NAP_PERIOD_SECONDS)

        rtcwake -d /dev/rtc0 -m standby -s $BATTERY_NAP_PERIOD_SECONDS
	# zzzzzzzzzzzzzzz......
        # sleeps inside rtcwake (until RTC alarm or LTC4156 interrupt or power key press)

        # Clear and disable LTC4156 interrupt
        # When an interrupt is pending, all the LTC4156 status registers are frozen
        i2cset -y 1 0x09 0x06 0x00

        log_battery after_suspend_with_rtc
	post_sleep_raw_soc=$raw_soc


        NOW_TIME=$(date +%s)
        EARLY_WAKEUP_TIME=$(expr $NAP_END_TIME - 5)

	# calculate average sleep current
	SLEEPING_TIME=$(expr $NOW_TIME - $SLEEP_START_TIME)
        echo Slept for $SLEEPING_TIME seconds
        raw_soc_used=$(expr $post_sleep_raw_soc - $pre_sleep_raw_soc)
        let battery_soc_uA_hour=$raw_soc_used\*446
        let average_current_mA=$battery_soc_uA_hour\*3600/$SLEEPING_TIME/1000
        echo Average current during sleep: $average_current_mA mA


        if [ $NOW_TIME -lt $EARLY_WAKEUP_TIME ]; then
            # Not a normal NAP, we woke up too early (at least 5 sec too early), due to external power or ON key press.
            # Either way, simply restart the application.
            LOOPING=1
            break
        fi
    done
    echo Finished suspend_with_rtc, LOOPING=$LOOPING
}

check_battery_soc()
{
   let raw_soc=`i2cget -y 1 0x70 0x02 w`
   if [ $raw_soc -ge 32768 ]
   then
       # Fix negative value
       let raw_soc=$raw_soc-65536
   fi
   let ram0=`i2cget -y 1 0x70 0x20 w`
   let raw_soc=$raw_soc+$ram0
   # Standard battery is 10 to 14 A.hr = 10000 to 14000 mA.hr
   # conversion factor is from STC3100 Datasheet, one LSB count is 6.70 uV.hr, 15 milli-ohm, 6.7/15=0.446
   let battery_soc_mA_hour=$raw_soc*446/1000
}

check_battery_current()
{
   let raw_current=`i2cget -y 1 0x70 0x06 w`
   #
   if [ $raw_current -ge 8192 ]
   then
       # Fix negative value
       let raw_current=$raw_current-16384
   fi

   # conversion factor is from STC3100 Datasheet, one LSB count is 11.77 uV, 15 milli-ohm, 11.77/15=0.785
   let battery_current_mA=$raw_current*785/1000
}

log_battery()
{
    # param 1 is reason
    DATESTAMP=`date +'%Y-%m-%d %H:%M:%S'`
    fast_check_battery_voltage
    detect_power
    REASON=$1
    check_battery_soc
    check_battery_current

    echo log_battery: $DATESTAMP,$millivolts,$battery_current_mA,$battery_soc_mA_hour,$POWER_SOURCE,$ram0,$REASON
    echo $DATESTAMP,$millivolts,$battery_current_mA,$battery_soc_mA_hour,$POWER_SOURCE,$ram0,$REASON >> /mnt/nand/logfiles/power.log
}

export HALT=1

export LOOPING=1
while [ $LOOPING -eq 1 ]; do
    cd /mnt/nand

    wait_for_eoc_and_check_battery_voltage
    if [ $millivolts -lt 2900 ]
    then
		zcat /etc/lowbattery_1024x600.ppm.gz | fbsplash -s -
		sleep 10
        echo Battery low.
        break
    fi

    log_battery before_running_app

    # Clear and disable LTC4156 interrupt
    # When an interrupt is pending, all the LTC4156 status registers are frozen
    i2cset -y 1 0x09 0x06 0x00

    # Turn on LCD power as we're going to display something
    echo 1 > /sys/class/gpio/gpio99/value
    usleep 100000
    echo 1 > /sys/class/gpio/gpio100/value

	# Limit size of power.log:
	if [ -e /mnt/nand/logfiles/power.log ]
	then
        if [ $(stat -c %s /mnt/nand/logfiles/power.log) -gt 100000 ]
        then
	        echo Rotating power.log...
	        rm -f /mnt/nand/logfiles/power.old.log
		    mv /mnt/nand/logfiles/power.log /mnt/nand/logfiles/power.old.log
        fi
	fi
	if [ -e /mnt/nand/logfiles/power.old.log ]
	then
        if [ $(stat -c %s /mnt/nand/logfiles/power.old.log) -gt 200000 ]
        then
	        echo Removing over-size logfile...
	        rm -f /mnt/nand/logfiles/power.old.log
        fi
	fi
	rm -f /tmp/logfiles/power*.log

    if [ -e /mnt/nand/logfiles ]; then
        cp -ar /mnt/nand/logfiles /tmp
    fi

    if [ -e /dev/crypto ]
    then
        # disable linux crypto hardware accelerators -  until EVM TI SDK fix this.
        rm /dev/crypto
    fi


    filePath="/proc/cmdline"
    BOOT_CMD_LINE=$(cat "$filePath")
    KYO_LCD="lcdmodel=KYO_TCG070WVLPEANN"
    echo $BOOT_CMD_LINE

    if echo $BOOT_CMD_LINE | grep -q $KYO_LCD ; then
        DISPLAY_RESOLUTION="800x480"
    else
        DISPLAY_RESOLUTION="1024x600"
    fi

    echo LCD DISPLAY_RESOLUTION=$DISPLAY_RESOLUTION

    LD_LIBRARY_PATH=/mnt/nand/Qt/lib ./PolarisQtApp -qws -nomouse -display "Multi: LinuxFB: VNC::size=$DISPLAY_RESOLUTION:depth=32:0" 2&> /dev/null
    # Halt Codes, see powersavings.h
    # 0 : Suspend: i.e. standby mode via rtcwake
    # 1 : Power cycle: i.e. do linux poweroff, which will shutdown if no external power, or restart if external power if present
    # 2 : Application restart
    # 3 : Factory reset
    # 4 : Application abort (reboot)
    HALT=$?

    # check if model has changed (e.g. upgrade).
    dd bs=1 skip=1024 count=12 < /sys/bus/i2c/devices/0-0050/eeprom 2>/dev/null | strings > /tmp/model
    # some early units has "model= XR500" instead of "model=XR5000", so just check for XR in first 12 characters
    if grep -qi XR /tmp/model
    then
        echo Detected XR5000 model
        ln -sf /etc/ttlogo_xr5000_${DISPLAY_RESOLUTION}_step_5.ppm.gz /etc/ttlogo_step_5.ppm.gz
    elif grep -qi ID /tmp/model
    then
        echo Detected ID5000 model
        ln -sf /etc/ttlogo_id5000_${DISPLAY_RESOLUTION}_step_5.ppm.gz /etc/ttlogo_step_5.ppm.gz
    else
        echo Detected JR5000 model
        ln -sf /etc/ttlogo_jr5000_${DISPLAY_RESOLUTION}_step_5.ppm.gz /etc/ttlogo_step_5.ppm.gz
    fi

    # show splash screen
    if [ -e /etc/ttlogo_step_5.ppm.gz ]; then
        zcat /etc/ttlogo_step_5.ppm.gz | fbsplash -s -
        # Simpler to just display screen for 2 seconds atleast!
        # And sleep here instead of outside the loop so the screen doesn't flash the logo
        sleep 2
    fi

    # remove old logfiles first because without this, if /tmp/logfiles/adc.log is deleted by user, nothing will be done on /mnt/nand/adc.log
    TMP_LOG_PATH=/mnt/nand/logfiles.old
    rm -rf $TMP_LOG_PATH && mv /mnt/nand/logfiles $TMP_LOG_PATH && cp -ar /tmp/logfiles /mnt/nand/logfiles

    # move new telemetry files to persistent storage
    mkdir -p /mnt/nand/telemetry
    mv /tmp/telemetry/*.log /mnt/nand/telemetry/
    # delete old telemetry files
    find /mnt/nand/telemetry -name "*.log" -type f -mtime +180 -delete

    wait_for_eoc_and_check_battery_voltage
    log_battery after_exit_app

    echo lun0 file: `cat /sys/bus/platform/devices/musb-hdrc.0/gadget/lun0/file`
    if grep -q disk.img /sys/bus/platform/devices/musb-hdrc.0/gadget/lun0/file
    then
        echo WARNING:  MSD USB gadget is attached, probably powered down while USB connected.
        echo Trying to eject MSD:
        echo > /sys/bus/platform/devices/musb-hdrc.0/gadget/lun0/file
        echo Disconnecting MSD gadget:
        rmmod g_mass_storage
        sh -x /mnt/nand/extract_ramdisk.sh > /mnt/nand/extract_ramdisk.log
    fi

    # Default is to shutdown.
    LOOPING=0

    if [ -e newdb ]; then
    # After USB to PC cable transfer, there may be a new database from the PC, so restart app.
        mv newdb db
        LOOPING=1
    fi

    if [ -e settings_current_new.ttfav ]; then
    # After USB to PC cable transfer, there may be a new settings from the PC, so restart app.
        mv settings_current_new.ttfav settings_current.ttfav
        LOOPING=1
    fi

    # firmware update - new method
    if [ -d update ]; then
        # After USB to PC cable transfer, there may be a new firmware from the PC, so restart app.
        cd /mnt/nand/update/
        pwd >> ../update.log
        if [ -e uImageRD ] && [ ! -e uImageRD-no-dtb ]; then
            echo "Pre version 4.0 update file, do not allow downgrade" >> ../update.log
            RC=0
        else
            chmod a+x update.sh
            ls /mnt/nand/update* >> ../update.log
            /mnt/nand/update/update.sh >> ../update.log
            RC=$?
        fi

        cd /mnt/nand/
        rm -rf /mnt/nand/update
        echo "Return code:$RC" >> update.log
        if [ $RC -eq 1 ]
        then
            echo "Rebooting because upgrade.sh needs it" >> update.log
            HALT=1
            FORCE_REBOOT=1
        else
            LOOPING=1
        fi
    fi

    if [ ! -e db ]; then
        # After reset to factory defaults, the database is deleted, so restart app.
        LOOPING=1
    fi

    detect_power

    # Disable suspend (standby) mode until resolved POL-512.
    # if [ $HALT -eq 0 ]
    # then
    #    HALT=1
    # fi

    # Avoid standby if BT looks broken.
    if [ -e /tmp/bt_broken ]
    then
        DATESTAMP=`date +'%Y-%m-%d %H:%M:%S'`
        echo $DATESTAMP,BT broken
        echo $DATESTAMP,Rebooting >> /mnt/nand/logfiles/power.log
        HALT=1
    fi

    if [ $HALT -eq 0 ]
    then
        if [ $LOOPING -eq 0 ]
        then
            # NB: bt.sh and wifi.sh shouldn't be running.
            # They appear to forcibly exit when the application shuts down.
	    if pidof bt.sh
            then
                echo bt.sh still running, shutting down.
                break
            fi
	    if pidof wifi.sh
            then
                echo wifi.sh still running, shutting down.
                break
            fi

            # Instead of powering down, do a suspend and then LOOP around again.
            LOOPING=1

            # S2 PCA:
            # GPIO 28 = power key input (0=pressed)
            # GPIO 63 = stable (1=on)
            # GPIO 96  = GPIO3_0 = LTC4156 IRQ
            # GPIO 99  = GPIO3_3 = LCD Power Enable (1=on)
            # GPIO 100 = GPIO3_4 = Backlight Power Enable (1=on)

            # Wake on power key press from standby fails unless we unexport this:
            echo 28 > /sys/class/gpio/unexport

            # Turn these off LCD power and stable LED
            echo 0 > /sys/class/gpio/gpio63/value
            echo 0 > /sys/class/gpio/gpio100/value
            usleep 100000
            echo 0 > /sys/class/gpio/gpio99/value
            # leave blue led on for debug for now.

            ifconfig wlan0 down
	    if (hciconfig hci0 | head -3 | tail -1 | grep -q UP)
            then
                # Disconnect BT connections first.
                for d in `hcitool -i hci0 con | tr " " "\n" | grep ..:..:..:..:..:..`
                do
                    hcitool -i hci0 dc $d
                done
                hciconfig hci0 down
            fi

            if [ -f /var/run/udhcpd_usb.pid ]
            then
                ifconfig usb0 down
                kill $(cat /var/run/udhcpd_usb.pid)
            fi
            rmmod g_ether
            rmmod g_mass_storage

            # umount /dev/sda1 if mounted (usb flash disk)
            grep -wq ^/dev/sda1 /proc/mounts && umount /dev/sda1

            # umount /dev/sda if mounted (usb flash disk without partition table)
            grep -wq ^/dev/sda /proc/mounts && umount /dev/sda

            rm /tmp/wifi_init_done

            # unsure all writes are completed
            sync

            if [ ! -d /debugfs ]; then mkdir -p /debugfs; mount -t debugfs debugfs /debugfs; fi

            detect_power
            if [ "$POWER_SOURCE" == "BATTERY" ]; then
                suspend_with_rtc $BATTERY_STANDBY_PERIOD

                # strangely the hci0 interface is UP after waking up again, even though it has been turned off before suspending.
                # so turn it off again, PolarisQtApp will enable it when it is ready.
                hciconfig hci0 down

                # check for stale usb disk /dev/sda*
                if [ -e /dev/sda ]; then
                    # if the read of /dev/sda fails, then remove the /dev/sda*
                    dd if=/dev/sda bs=512 count=1 2> /dev/null > /dev/null || rm /dev/sda*
                fi
            fi
        fi
    elif [ $HALT -eq 1 ]
    then
        LOOPING=0
    elif [ $HALT -eq 2 ]
    then
        LOOPING=1
    elif [ $HALT -eq 3 ]
    then
        # Factory reset
        LOOPING=1
        rm db
        rm favourites/*
        rm *.ttfav
        cp wifi/wpa_supplication.conf_default wifi/wpa_supplicant.conf
    elif [ $HALT -eq 4 ]
    then
        LOOPING=0
    fi
done

wait_for_eoc_and_check_battery_voltage
log_battery before_shutdown

cd /
umount /dev/loop0
losetup -d /dev/loop0
sync

if [ $FORCE_REBOOT -eq 1 ]
then
    echo Doing reboot...
    reboot
fi

echo Doing poweroff...
poweroff
